<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;
use yii\db\Expression;

class User extends ActiveRecord implements IdentityInterface
{
    const SCENARIO_REGISTER = 'register';

    public $password;
    public $password_repeat;
    public $agree_rules;

    public static function tableName()
    {
        return 'user';
    }

    public function behaviors()
    {
        return [
            [
                'class' => \yii\behaviors\TimestampBehavior::class,
                'createdAtAttribute' => 'created_at',
                'updatedAtAttribute' => false,
                'value' => new Expression('NOW()'), // Используем SQL-функцию NOW()
            ],
        ];
    }

    public function scenarios()
    {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_REGISTER] = [
            'email', 'password', 'password_repeat', 'fio', 'phone',
            'passport', 'address', 'agree_rules'
        ];
        return $scenarios;
    }

    public function rules()
    {
        return [
            // Обязательные поля при регистрации
            [['email', 'password', 'password_repeat', 'fio', 'phone', 'passport', 'address', 'agree_rules'], 'required', 'on' => self::SCENARIO_REGISTER],

            // Email
            ['email', 'email'],
            ['email', 'unique', 'message' => 'Этот email уже используется'],

            // Пароли
            ['password', 'string', 'min' => 7, 'max' => 255],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают'],

            // ФИО
            ['fio', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s\-]+$/u', 'message' => 'Только кириллица, пробелы и дефисы'],

            // Телефон - упрощаем формат
            ['phone', 'match', 'pattern' => '/^\+7\d{10}$/', 'message' => 'Формат: +7XXXXXXXXXX (11 цифр)'],

            // Паспорт
            ['passport', 'match', 'pattern' => '/^\d{10}$/', 'message' => '10 цифр без пробелов'],

            // Согласие
            ['agree_rules', 'compare', 'compareValue' => 1, 'message' => 'Необходимо согласиться с правилами', 'on' => self::SCENARIO_REGISTER],

            // Роль по умолчанию
            ['role', 'default', 'value' => 'user'],
            ['role', 'in', 'range' => ['user', 'admin']],
        ];
    }

    public function attributeLabels()
    {
        return [
            'email' => 'Email',
            'password' => 'Пароль',
            'password_repeat' => 'Повторите пароль',
            'fio' => 'ФИО',
            'phone' => 'Телефон',
            'passport' => 'Паспорт (10 цифр)',
            'address' => 'Адрес',
            'agree_rules' => 'Согласен с правилами',
        ];
    }

    // IdentityInterface методы
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAuthKey()
    {
        return null;
    }

    public function validateAuthKey($authKey)
    {
        return false;
    }

    public static function findByEmail($email)
    {
        return static::findOne(['email' => $email]);
    }

    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                // Устанавливаем пароль только для новых записей
                if (!empty($this->password)) {
                    $this->setPassword($this->password);
                }
                // Устанавливаем роль по умолчанию
                if (empty($this->role)) {
                    $this->role = 'user';
                }
                // Устанавливаем текущую дату и время, если поведение не сработало
                if (empty($this->created_at)) {
                    $this->created_at = date('Y-m-d H:i:s');
                }
            }
            return true;
        }
        return false;
    }
}